/* ROCK Linux Wrapper for getting a list of created files

   gcc -Wall -O2 -ldl --shared -o fl_wrapper.so fl_wrapper.c

   ELF Dynamic Loading Documentation:
    - http://www.linuxdoc.org/HOWTO/GCC-HOWTO-7.html
    - http://www.educ.umu.se/~bjorn/mhonarc-files/linux-gcc/msg00576.html
    - /usr/include/dlfcn.h
*/


/* Headers and prototypes */

#define _GNU_SOURCE
#define _REENTRANT

#ifdef DEBUG_ME
#  define debug_fprintf fprintf
#else
#  define debug_fprintf(args...)
#endif

#ifndef NEW_FILES_LOG
#  define NEW_FILES_LOG "new_files_log.txt"
#endif

#define open xxx_open
#define open64 xxx_open64
#  include <dlfcn.h>
#  include <errno.h>
#  include <fcntl.h>
#  include <stdio.h>
#  include <stdlib.h>
#  include <string.h>
#  include <sys/stat.h>
//  #  include <sys/time.h>
#  include <sys/types.h>
#  include <unistd.h>
#  include <utime.h>
#undef open
#undef open64

void * get_dl_symbol(char *);

struct status_t {
	ino_t   inode;
	off_t   size;
	time_t  mtime;
	time_t  ctime;
};

void handle_file_access_before(const char *, const char *, struct status_t *);
void handle_file_access_after(const char *, const char *, struct status_t *);

/* Wrapper Functions */

#define wrfunc3(nr,rt,name,arg1,arg2,arg3)				\
extern rt name(arg1, arg2, arg3);					\
rt (*orig_ ## name)(arg1, arg2, arg3) = NULL;				\
rt name(arg1 a1, arg2 a2, arg3 a3) {					\
	rt rc; struct status_t status; int old_errno=errno;		\
	handle_file_access_before(#name,a ## nr, &status);		\
	if (!orig_ ## name) orig_ ## name=get_dl_symbol(#name);	\
	errno=old_errno; rc=orig_ ## name(a1,a2,a3); old_errno=errno;	\
	handle_file_access_after(#name, a ## nr, &status);		\
	errno=old_errno; return rc;					\
}

#define wrfunc2(nr,rt,name,arg1,arg2)					\
extern rt name(arg1, arg2);						\
rt (*orig_ ## name)(arg1, arg2) = NULL;					\
rt name(arg1 a1, arg2 a2) {						\
	rt rc; struct status_t status; int old_errno=errno;		\
	handle_file_access_before(#name,a ## nr, &status);		\
	if (!orig_ ## name) orig_ ## name=get_dl_symbol(#name);	\
	errno=old_errno; rc=orig_ ## name(a1,a2); old_errno=errno;	\
	handle_file_access_after(#name, a ## nr, &status);		\
	errno=old_errno; return rc;					\
}

wrfunc3(1, int, open,   const char *, int, int)
wrfunc3(1, int, open64, const char *, int, int)

wrfunc2(1, FILE *, fopen,   const char *, const char *)
wrfunc2(1, FILE *, fopen64, const char *, const char *)

wrfunc2(1, int, creat,   const char *, mode_t)
wrfunc2(1, int, creat64, const char *, mode_t)

wrfunc2(1, int, mkdir, const char *, mode_t)
wrfunc3(1, int, mknod, const char *, mode_t, dev_t)

wrfunc2(2, int, link,    const char *, const char *)
wrfunc2(2, int, symlink, const char *, const char *)
wrfunc2(2, int, rename,  const char *, const char *)

wrfunc2(1, int, utime,  const char *, const struct utimbuf *)
wrfunc2(1, int, utimes, const char *, struct timeval *)

#if 0
/* This hack is needed for glibc 2.1.1 (and others?) */
extern int fclose(FILE *);
int (*orig_fclose)(FILE *) = NULL;
int fclose(FILE * f) {
	int old_errno=errno;
	if (!orig_fclose) orig_fclose=get_dl_symbol("fclose");
	old_errno=errno; return orig_fclose(f);
}
#endif


/* Internal Functions */

void * get_dl_symbol(char * symname) {
        void * rc = dlsym(RTLD_NEXT,symname);
	if (!rc) {
		printf("\n>> ROCK Linux Wrapper Lib: Can't "
		       "resolve %s: %s<<\n\n",symname,dlerror());
		abort();
	}
        return rc;
}

void handle_file_access_before(const char * func, const char * file,
                               struct status_t * status) {
	struct stat st;
	debug_fprintf(stderr,"+ %s: %s (%d)\n",func,file,lstat(file,&st));
	if ( lstat(file,&st) ) {
		status->inode=0;  status->size=0;
		status->mtime=0;  status->ctime=0;
	} else {
		status->inode=st.st_ino;    status->size=st.st_size;
		status->mtime=st.st_mtime;  status->ctime=st.st_ctime;
	}
}

void handle_file_access_after(const char * func, const char * file,
                              struct status_t * status) {
	int fd; char buf[512],*buf2; struct stat st;
	if ( strcmp(file,NEW_FILES_LOG) && !lstat(file,&st) &&
             (status->inode != st.st_ino   || status->size  != st.st_size ||
              status->mtime != st.st_mtime || status->ctime != st.st_ctime) ) {
		fd=open(NEW_FILES_LOG,O_APPEND|O_WRONLY,0);
		if (fd == -1) return;
		if (file[0] == '/') {
			sprintf(buf,"%s:%*s%s\n",func,
			        (int)(10-strlen(func)),"",file);
		} else {
			buf2=get_current_dir_name();
			sprintf(buf,"%s:%*s%s/%s\n",func,
			        (int)(10-strlen(func)),"",buf2,file);
			free(buf2);
		}
		write(fd,buf,strlen(buf));
		close(fd);
	}
	debug_fprintf(stderr,"- %s: %s (%d)\n",func,file,lstat(file,&st));
}
