/*
    ROCK Linux Install Shell
    Copyright (C) 1998, 1999, 2000, 2001, 2002
        Clifford Wolf, Valentin Ziegler and Rene Rebe

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
 * ChangeLog:
 * 2002-06-19: Ren Rebe
 *   * started this file
 *   * validated every open call to check the return value
 *   * added error messages to the failing open calls
 */

#ifndef ROCKVER
#  define ROCKVER "0000"
#endif

#define _GNU_SOURCE

#ifndef USE_READLINE
#  define USE_READLINE 0
#endif

#define PKG_NUM 2000

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/vfs.h>
#include <fcntl.h>
#include <fnmatch.h>
#include <string.h>
#include <sys/time.h>
#include <dirent.h>
#include <stdarg.h>
#include <signal.h>
#include <setjmp.h>

#if USE_READLINE == 1
#  include <readline/readline.h>
#  include <readline/history.h>
#endif

struct ext_package {
	char name[100];
	char folder[100];
	char imported;
};

struct package {
	char name[100];
	char folder[100];
	char fullname[200];
	char filename[100];
	int size;
	int selected;
};

#if 0
char display[][11]={
  "::::::####",  "#::::::###",  "##::::::##",  "###::::::#",
  "####::::::",  "#####:::::",  "######::::",  ":######:::",
  "::######::",  ":::######:",  "::::######",  ":::::#####"};
#endif

#if 1
char display[][11]={
  "ROCK.Linux",  "OCK.Linux.",  "CK.Linux.I",  "K.Linux.In",
  ".Linux.Ins",  "Linux.Inst",  "inux.Insta",  "nux.Instal",
  "ux.Install",  "x.Install.",  ".Install.S",  "Install.Sh",
  "nstall.She",  "stall.Shel",  "tall.Shell",  "all.Shell.",
  "ll.Shell..",  "..Shell...",  ".Shell....",  "Shell....R",
  "hell....R.",  "ell....R..",  "ll....R...",  "l....R...O",
  "....R...O.",  "...R...O..",  "..R...O...",  ".R...O...C",
  "R...O...C.",  "R..O...C..",  "R.O...C...",  "RO...C...K",
  "RO..C...K.",  "RO.C...K..",  "ROC...K...",  "ROC..K...L",
  "ROC.K...L.",  "ROCK...L..",  "ROCK..L...",  "ROCK.L...i",
  "ROCK.L..i.",  "ROCK.L.i..",  "ROCK.Li...",  "ROCK.Li..n",
  "ROCK.Li.n.",  "ROCK.Lin..",  "ROCK.Linu.",  "ROCK.Linux"};
#endif

#if 0
char display[][11]={
  "ROCK.Linux",  "ROCK.Linux",  "ROCK.Linux",  "ROCK.Linux",
  "ROCK.Linux",  ".ROCK.Linu",  "..ROCK.Lin",  "...ROCK.Li",
  ".t..ROCK.L",  ".ta..ROCK.",  ".sta..ROCK",  ".stal..ROC",
  ".nstal..RO",  ".nstall..R",  ".Install..",  ".Install..",
  ".Install..",  "Install..S",  "nstall..Sh",  "stall..She",
  "tall..Shel",  "all..Shell",  "ll..Shell.",  "l..Shell..",
  "..Shell...",  ".Shell....",  "Shell....R",  "hell....R.",
  "ell....R..",  "ll....R...",  "l....R...O",  "....R...O.",
  "...R...O..",  "..R...O...",  ".R...O...C",  "R...O...C.",
  "R..O...C..",  "R.O...C...",  "RO...C...K",  "RO..C...K.",
  "RO.C...K..",  "ROC...K...",  "ROC..K...L",  "ROC.K...L.",
  "ROCK...L..",  "ROCK..L...",  "ROCK.L...i",  "ROCK.L..i.",
  "ROCK.L.i..",  "ROCK.Li...",  "ROCK.Li..n",  "ROCK.Li.n.",
  "ROCK.Lin..",  "ROCK.Linu.",  "ROCK.Linux",  "ROCK.Linux",
  "ROCK.Linux",  "OCK.Linux.",  "CK.Linux.I",  "K.Linux.In",
  ".Linux.Ins",  "Linux.Inst",  "inux.Insta",  "nux.Instal",
  "ux.Install",  "x.Install.",  ".Install.S",  "Install.Sh",
  "nstall.She",  "stall.Shel",  "tall.Shell",  "all.Shell.",
  "ll.Shell..",  "l.Shell...",  ".Shell....",  "Shell.....",
  "hell......",  "ell.......",  "ll........",  "l....L....",
  "...K.Li...",  "..CK.Lin..",  ".OCK.Linu.",  "ROCK.Linux"};
#endif

#define DISPLAY_NR (sizeof(display)/11)

typedef struct package package;
typedef struct ext_package ext_package;

package pkgs[PKG_NUM];
ext_package epkgs[PKG_NUM];

int sortbyname=1;
int pkg_num,epkg_num;

jmp_buf crtl_c_goto;
int     pg_size = 0;
int     pg_end  = 0;
int     pg_max  = 24;
char *  pg_text = NULL;

void crtl_c_handler(int x) {
	printf("\n"); fflush(stdout);
	longjmp(crtl_c_goto,1);
}

void pg_ready() {
	struct sigaction act;
	static int lines,c,ch; /* static, so it can't be clobbered */
	lines=c=0;             /* by the longjmp()                 */
	if (!isatty(0) || !isatty(1) || pg_max<2)
		{ if (pg_end) puts(pg_text); pg_end=0; return; }
	act.sa_handler = crtl_c_handler; sigemptyset(&act.sa_mask);
	act.sa_flags = SA_NOMASK; sigaction(SIGINT,&act,NULL);
	if (!setjmp(crtl_c_goto))
	    while (c < pg_end) {
		fputc(pg_text[c],stdout);  if (pg_text[c] == '\n') lines++;
		if (pg_text[c++] == '\n' && lines%pg_max == 0) {
			printf("--- MORE: Press enter ---"); fflush(stdout);
			while ( (ch=fgetc(stdin)) ) if (ch == '\n') break;
		}
	    }
	signal(SIGINT,SIG_IGN); pg_end=0;
}

int ctrl_c_sleep(int sec) {
	struct sigaction act;

	act.sa_handler = crtl_c_handler; sigemptyset(&act.sa_mask);
	act.sa_flags = SA_NOMASK; sigaction(SIGINT,&act,NULL);

	if (!setjmp(crtl_c_goto)) {
		sleep(sec); signal(SIGINT,SIG_IGN); return 0;
	}

	signal(SIGINT,SIG_IGN); return 1;
}

/*  This printf implements a little pager. The text is stored in the pg_text
 *  variable. It will be printed by pg_ready().
 */
int pg_printf (const char *format, ...)
{
	va_list arg;
	int done;
	
	if (pg_size-pg_end < 4500 || pg_text == NULL)
		pg_text=realloc(pg_text,pg_size=(pg_end+5000));
	
	va_start (arg, format);
	done = vsnprintf(pg_text+pg_end, 4000, format, arg);
	va_end (arg);
	
	if (done > 0) pg_end+=done>4000?4000:done;
	
	return done;
}

void usage(char * s)
{
	printf("%sUsage: install source-dir destination-dir\n",s);
	exit(1);
}

void help()
{
	printf("\n"
"  ROCK Linux Install Shell Commands:\n"
"\n"
"list [pattern]  ...........  List all packages\n"
"list-(un)sel [pattern]  ...  List (not) selected packages\n"
"list-ext [pattern]  .......  List not-imported extensions\n"
"list-cat  .................  List all cathegories\n"
"sort-name, sort-size  .....  Change package order in list\n"
"\n"
"(un)select [pattern]  .....  (Un)select the given package\n"
"(un)import [pattern]  .....  (Un)import the given extension package\n"
"\n"
"info pkg-name  ....  Print package info\n"
"files pkg-name  ...  Print package file listing\n"
"find [pattern]  ...  Find a file in all packages\n"
"sum  ..............  Print config summary\n"
"df  ...............  Print disk usage summary\n"
"\n"
"quit, q, exit  ....  Exit without install (Ctrl-D is also possible)\n"
"install, inst  ....  Install and exit\n"
"\n"
"pager n  ..........  Set the pager to n lines per screen (0=off, default=25)\n"
"\n"
"Note: 'pattern' is a shell wildcard pattern.\n"
"\n");
}

int epkgs_sort_name(const void * x, const void * y)
{
	const ext_package *a=x, *b=y;
	char a_fullname[200], b_fullname[200];
	sprintf(a_fullname,"%s/%s",a->folder,a->name);
	sprintf(b_fullname,"%s/%s",b->folder,b->name);
	return strcmp(a_fullname,b_fullname);
}

int pkgs_sort_name(const void * x, const void * y)
{
	const package *a=x, *b=y;
	return strcmp(a->fullname,b->fullname);
}

int pkgs_sort_size(const void * x, const void * y)
{
	const package *a=x, *b=y;
	return a->size < b->size;
}

int fnmatch2(char *pattern, const char *string, int flags)
{
	char *p1, *p2;
	int len, rc;

	while ( (p1=strchr(pattern,'\t')) != NULL ) *p1=' ';

	for (p1=pattern; p1 && *p1; p1=strchr(p1,' ')) {
		while (*p1 == ' ') p1++;
		if ( ! *p1 ) break;
		if ( (p2=strchr(p1,' ')) == NULL ) len=strlen(p1);
		else len=p2-p1;
		p2=malloc(len+1); memcpy(p2,p1,len); p2[len]=0;
		rc=fnmatch(p2, string, flags); free(p2);
		if ( ! rc ) return 0;
	}
	return 1;
}

int matchpkg(int pkgnr, char * expr)
{
	char name[200];
	if (!expr[0]) return 1;
	sprintf(name,"%s/%s",pkgs[pkgnr].folder,pkgs[pkgnr].name);
	return !fnmatch2(expr,name,FNM_CASEFOLD|FNM_LEADING_DIR) ||
	       !fnmatch2(expr,pkgs[pkgnr].name,FNM_CASEFOLD);
}

void list(int sel, char * expr)
{
	int c,c2;
	char name[200];
	
	for (c=c2=0; c<pkg_num; c++) {
		if (sel>=0 && sel!=pkgs[c].selected) continue;
		if (!matchpkg(c,expr)) continue;
		sprintf(name,"%s/%s",pkgs[c].folder,pkgs[c].name);
		c2++;
		if (c2%2==0) pg_printf("%c %5.1fM %s\n",pkgs[c].selected?'x':'o',pkgs[c].size/1024.0,name);
		else pg_printf("%c %5.1fM %-30s",pkgs[c].selected?'x':'o',pkgs[c].size/1024.0,name);
	}
	if (c2%2) pg_printf("\n");
}

void listcat()
{
	int c,c2;
	char oldcat[200]="";
	char output[200];
	
	if (!sortbyname) qsort(pkgs,pkg_num,sizeof(package),&pkgs_sort_name);
	for (c=c2=0; c<pkg_num; c++) {
		if (!strcmp(oldcat,pkgs[c].folder)) continue;
		c2++;
		sprintf(output,"%s/*",pkgs[c].folder);
		strcpy(oldcat,pkgs[c].folder);
		if (c2%5==0) pg_printf("%s\n",output);
		else pg_printf("%-15s",output);
	}
	if (c2%5) pg_printf("\n");
	if (!sortbyname) qsort(pkgs,pkg_num,sizeof(package),&pkgs_sort_size);
}

void listext(char * expr)
{
	int c,c2;
	char output[200];

	for (c=c2=0; c<epkg_num; c++) {
		sprintf(output,"%s/%s",epkgs[c].folder,epkgs[c].name);
		if ( epkgs[c].imported || ( expr[0] &&
			fnmatch2(expr,output,
			FNM_CASEFOLD|FNM_LEADING_DIR) ) ) continue;
		if (++c2%2==0) pg_printf("%s\n",output);
		else pg_printf("%-38s",output);
	}
	if (c2%2) pg_printf("\n");
}

void sel_unsel(int mode, char * expr)
{
	int c,c2;
	char name[200];
	
	for (c=c2=0; c<pkg_num; c++) {
		if (!matchpkg(c,expr)) continue;
		sprintf(name,"%s/%s",pkgs[c].folder,pkgs[c].name);
		pkgs[c].selected=mode;
		c2++;
		if (c2%2==0) pg_printf("%c %5.1fM %s\n",pkgs[c].selected?'x':'o',pkgs[c].size/1024.0,name);
		else pg_printf("%c %5.1fM %-30s",pkgs[c].selected?'x':'o',pkgs[c].size/1024.0,name);
	}
	if (c2%2) pg_printf("\n");
} 

void pkg_info(char * name, char * subdir, char * infodir)
{
	char fn[500];
	int fd,rc;
	
	sprintf(fn,"%s/%s/%s",infodir,subdir,name);
	if ( (fd=open(fn,O_RDONLY)) < 0 ) {
		printf("ERROR: No such package found.\n");
	} else {
		fflush(stdout);
		while ( (rc=read(fd,fn,500)) > 0 ) {
			write(1,fn,rc);
		}
		close(fd);
	}
}

int pkg_size(char * name, char * infodir)
{
	char fn[500];
	float rc=0;
	FILE *f;
	
	sprintf(fn,"%s/packages/%s",infodir,name);
	if ( (f=fopen(fn,"r")) != NULL ) {
		fn[499]=0;
		while ( fgets(fn,499,f) != NULL ) {
			if ( sscanf(fn,"Package Size: %f MB, %*d files",&rc) > 0 ) break;
		}
		fclose(f);
	}
	
	return (int)(rc*1024);
}

void inst_sum(char * basedir, char * infodir, char * instdir)
{
	int sel=0,imp=0,kb=0,c;
	
	for (c=0; c<pkg_num; c++) {
		if (!pkgs[c].selected) continue;
		sel++; kb+=pkg_size(pkgs[c].name,infodir);
	}
	
	for (c=0; c<epkg_num; c++) {
		if (epkgs[c].imported) imp++;
	}
	
	printf("Info dir:    %s\n",infodir);
	printf("Source dir:  %s\n",basedir);
	printf("Target dir:  %s\n",instdir);
	printf("%d/%d packages selected (%d MB).\n",
	       sel,pkg_num,kb/1024);
	printf("%d/%d extensions imported.\n",imp,epkg_num);
}

int pkg_files(char * name, char * infodir)
{
	char fn[500];
	int rc=100;
	FILE *f;
	
	sprintf(fn,"%s/packages/%s",infodir,name);
	if ( (f=fopen(fn,"r")) != NULL ) {
		fn[499]=0;
		while ( fgets(fn,499,f) != NULL ) {
			if ( sscanf(fn,"Package Size: %*f MB, %d files",&rc) > 0 ) break;
		}
		fclose(f);
	}
	return rc;
}

void find_file(char * src_expr, char * infodir)
{
	int c;
	char expr[200];
	char fn[500];
	FILE *f;
	
	sprintf(expr,"*: %s",src_expr);
	
	for (c=0; c<pkg_num; c++) {
	  sprintf(fn,"%s/flists/%s",infodir,pkgs[c].name);
	  if ( (f=fopen(fn,"r")) != NULL ) {
	    fn[499]=0;
	    while ( fgets(fn,499,f) != NULL ) {
	    	if (fn[0] && fn[strlen(fn)-1]=='\n')
	    		fn[strlen(fn)-1]=0;
		if (!fnmatch2(expr,fn,FNM_CASEFOLD))
			pg_printf("%s\n",fn);
	    }
	    fclose(f);
	  }
	}
}

void install(char * basedir, char * instdir, char * infodir)
{
	int c,c2,c3,c4,c5,files,sel,dotsc;
	struct timeval last_tv,this_tv;
	char name[200],cmd[500],dots[100];
	FILE *p;
	
	for (c=sel=0; c<pkg_num; c++)
		if (pkgs[c].selected) sel++;
	
	gettimeofday(&last_tv,NULL);
	for (c=c2=c4=0; c<pkg_num; c++) {
		if (!pkgs[c].selected) continue;
		sprintf(name,"%s/%s",pkgs[c].folder,pkgs[c].name);
		files=pkg_files(pkgs[c].name,infodir);
		c2++; dotsc=27-strlen(name);
		sprintf(dots,"%.*s",dotsc,"......................."
			".........................................");
		if (isatty(1))
		    printf("%03d%% (%03d/%03d) Installing %s ...%s.. 000%% [%s] -",
		           (c2*100)/sel,c2,sel,name,dots,display[c4%DISPLAY_NR]);
		else {
		    printf("%03d%% (%03d/%03d) Installing %s ...",
		           (c2*100)/sel,c2,sel,name);
		    if (files < 10)    { strcat(dots,"."); dotsc++; }
		    if (files < 100)   { strcat(dots,"."); dotsc++; }
		    if (files < 1000)  { strcat(dots,"."); dotsc++; }
		    if (files < 10000) { strcat(dots,"."); dotsc++; }
		}
		fflush(stdout);
		if ( !strncmp(basedir,"ftp://",6) ||
		     !strncmp(basedir,"http://",7) )
			sprintf(cmd,"wget -q -O - %s/%s | "
			            "tar xvf - -C %s --same-owner "
			            "--use-compress-program=bzip2 -p",
			            basedir,pkgs[c].filename,instdir);
		else
			sprintf(cmd,"tar xvf %s/%s -C %s "
			            "--use-compress-program=bzip2 "
			            "--same-owner --same-permissions",
			            basedir,pkgs[c].filename,instdir);
		p=popen(cmd,"r");
		for (c5=0, c3=1; fgets(cmd,499,p) != NULL; c3++) {
			if (isatty(1)) {
			    printf("\r%03d%% (%03d/%03d) Installing %s ..."
			           "%s...%3d%% [%s] %c",(c2*100)/sel,
			           c2,sel,name,dots,(c3*100)/files,
			           display[c4%DISPLAY_NR],"-\\|/"[c3%4]);
			} else {
			    if ( c5 < (c3*dotsc)/files && c5 < dotsc)
				{ printf("."); c5++; }
			}
			gettimeofday(&this_tv,NULL);
			if (this_tv.tv_sec > last_tv.tv_sec ||
			    this_tv.tv_usec > last_tv.tv_usec+100000) {
				last_tv=this_tv; c4++;  
			}
			fflush(stdout);
		}
		if ( pclose(p) ) {
			printf("\n
      FATAL ERROR:   Installation of package '%s' failed!

This could have multiple resons:   If you are installing  +----------------+
over a  network,  the link  to your  ftp or  http server  |  \\|/ ____ \\|/  |
could be down - or the server itself. It's also possible  |  \"@'/ ,. \\`@\"  |
that the server reached it's  maximum number of connect-  |  /_| \\__/ |_\\  |
ions.  Your hard disk  may be full or  the  installation  |     \\__U_/     |
media might be broken. Also a memory problem, some other  |      |  |      |
hardware error or a software bug are possible reasons.    +----------------+

Press Ctrl-C to skip this package and continue the installation ...

I'll retry to install the package in 30 seconds ...\n",name);

			if (!ctrl_c_sleep(30)) c--;
			continue;
		}
		if (isatty(1)) {
			if (files < 10)    { strcat(dots,"."); dotsc++; }
			if (files < 100)   { strcat(dots,"."); dotsc++; }
			if (files < 1000)  { strcat(dots,"."); dotsc++; }
			if (files < 10000) { strcat(dots,"."); dotsc++; }
			printf("\r%03d%% (%03d/%03d) Installing %s ...%s "
			       "%d files installed\n",(c2*100)/sel,c2,
			       sel,name,dots,files);
		} else
			printf("%s %d files installed\n",dots+c5,files);
		fflush(stdout);
	}
	printf("%d Packages installed.%s\n",c2,isatty(1)?"\007":"");
}

struct df_ent {
	char path[200];
	long long dfe_b_all;
	long long dfe_b_free;
	long long dfe_b_pkgs;
	long long dfe_i_all;
	long long dfe_i_free;
	long long dfe_i_pkgs;
};

void print_df(char * infodir, char * instdir)
{
	FILE *f;
	char fn[400],tmp[400];
	struct statfs stfs;
	struct df_ent df_list[100];
	int dfl_count=0;
	int c,c2,i,size;
	
	printf("Mount Point                         Size  "
	/*      /______________________________123456 MB__   */
	       "     Free now     Free after install\n");
	/*      123456_MB_(123_%)__123456_MB_(123_%)         */

	/* Get a the list of filesystems */
	if ( (f=fopen("/proc/mounts","rt")) == NULL) {
	  printf("ERROR: /proc/mounts not found - no FS statistic possible\n");
	}
	else
	  {
	  while ( fscanf(f,"%*s %s %*s %*s %*d %*d\n",fn) == 1 )
	    if ( ! statfs(fn,&stfs) ) {
	      stfs.f_bfree = stfs.f_bavail ;
	      
	      strcpy(df_list[dfl_count].path,fn);
	      df_list[dfl_count].dfe_b_all  =
		(long long)stfs.f_blocks * (long long)stfs.f_bsize + 1;
	      df_list[dfl_count].dfe_b_free =
		(long long)stfs.f_bfree  * (long long)stfs.f_bsize ;
	      df_list[dfl_count].dfe_b_pkgs = 0 ;
	      
	      df_list[dfl_count].dfe_i_all  = (long long)stfs.f_files + 1 ;
	      df_list[dfl_count].dfe_i_free = (long long)stfs.f_ffree ;
	      df_list[dfl_count].dfe_i_pkgs = 0 ;
	      
	      dfl_count++;
	    }
	  fclose(f);
	}

	/* Accumulate the packages file-sizes */
	for (c=0; c<pkg_num; c++) {
	  if (!pkgs[c].selected) continue;
	  sprintf(fn,"%s/cksums/%s",infodir,pkgs[c].name);
	  if ( (f=fopen(fn,"rt")) == NULL) {
	    printf("ERROR: cksum file %s not found!\n",fn);
	    return;
	  }
	  while ( fscanf(f,"%*d %d %s\n",&size,tmp) == 2 ) {
	    sprintf(fn,"%s/%s",instdir,tmp);
	    for (c2=i=0; c2<dfl_count; c2++) {
	      sprintf(tmp,"%s/",df_list[c2].path);
	      if (!strncmp(fn,tmp,strlen(tmp))) i=c2;
	    }
	    df_list[i].dfe_b_pkgs += size;
	    df_list[i].dfe_i_pkgs++;
	  }
	  fclose(f);
	}

	for (c=0; c<dfl_count; c++)
	  if (df_list[c].dfe_i_pkgs)
	    printf("%-30s %6Ld MB  %6Ld MB (%3.0f %%)  %6Ld MB (%3.0f %%)\n",
		   df_list[c].path, (df_list[c].dfe_b_all + 524288) / 1048576,
		   (df_list[c].dfe_b_free + 524288) / 1048576,
		   100.0 * ((float)df_list[c].dfe_b_free / df_list[c].dfe_b_all),
		   (df_list[c].dfe_b_free - df_list[c].dfe_b_pkgs + 524288) / 1048576,
		   100.0 * ((float)(df_list[c].dfe_b_free - df_list[c].dfe_b_pkgs) / df_list[c].dfe_b_all)
		   );
}

void post_install(char * inst_dir)
{
	FILE *f; char fn[256],cmd[256];
	
	sprintf(fn,"%s/run_setup.sh",inst_dir);
	if ( (f=fopen(fn,"wt")) == NULL)
	  {
	    printf("ERROR: Can't write run_setup script!\n");
	  }
	else
	  {
	    fprintf(f,". /etc/profile ; grep ' %s' /etc/mtab.in | \\\n"
		    "sed 's, %s/\\?, /,' > /etc/mtab ; rm -f /etc/mtab.in\n"
		    "echo\n" "for x in `ls /etc/setup.d/*` ; do\n"
		    "  [ -f $x -a -x $x ] && $x setup\n"
		    "done\n" "umount /dev /proc\n", inst_dir, inst_dir);
	    fclose(f); sprintf(cmd,"cd %s ; cat /proc/mounts > etc/mtab.in ; "
			       "chroot . bin/sh run_setup.sh", inst_dir);
	    system(cmd); unlink(fn);
	  }
}

void ext_unimport(char * expr)
{
	int c,c2,c3;
	char output[200];

	for (c=c2=0; c<epkg_num; c++) {
		if ( !epkgs[c].imported ) continue;
		sprintf(output,"%s/%s",epkgs[c].folder,epkgs[c].name);
		if ( expr[0] && fnmatch2(expr,output,
                     FNM_CASEFOLD|FNM_LEADING_DIR) &&
		     fnmatch2(expr,epkgs[c].name,
		     FNM_CASEFOLD|FNM_LEADING_DIR) ) continue;
		epkgs[c].imported=0;
		for (c3=0; c3<pkg_num; c3++)
		    if (!strcmp(pkgs[c3].name,epkgs[c].name)) {
			if (c3 != pkg_num-1)
			    memcpy(&pkgs[c3],&pkgs[pkg_num-1],sizeof(package));
			pkg_num--;
		    }
		if (++c2%2==0) pg_printf("%s\n",output);
		else pg_printf("%-38s",output);
	}
	if (c2%2) pg_printf("\n");

	if (sortbyname) qsort(pkgs,pkg_num,sizeof(package),&pkgs_sort_name);
	else qsort(pkgs,pkg_num,sizeof(package),&pkgs_sort_size);
}

void ext_import(char * expr, char * infodir)
{
	int c,c2;
	char output[200];

	for (c=c2=0; c<epkg_num; c++) {
		if ( epkgs[c].imported ) continue;
		sprintf(output,"%s/%s",epkgs[c].folder,epkgs[c].name);
		if ( expr[0] && fnmatch2(expr,output,
                     FNM_CASEFOLD|FNM_LEADING_DIR) &&
		     fnmatch2(expr,epkgs[c].name,
		     FNM_CASEFOLD|FNM_LEADING_DIR) ) continue;
		epkgs[c].imported=1;
		strcpy(pkgs[pkg_num].name,epkgs[c].name);
		strcpy(pkgs[pkg_num].folder,"ext");
		pkgs[pkg_num].selected=0;
		pkgs[pkg_num].size=pkg_size(pkgs[pkg_num].name,infodir);
		sprintf(pkgs[pkg_num].fullname,"ext/%s",pkgs[pkg_num].name);
		sprintf(pkgs[pkg_num].filename,"ext-pkgs/%s/%s/%s.tar.bz2",
			epkgs[c].folder,pkgs[pkg_num].name,pkgs[pkg_num].name);
		pkg_num++;
		if (++c2%2==0) pg_printf("%s\n",output);
		else pg_printf("%-38s",output);
	}
	if (c2%2) pg_printf("\n");

	if (sortbyname) qsort(pkgs,pkg_num,sizeof(package),&pkgs_sort_name);
	else qsort(pkgs,pkg_num,sizeof(package),&pkgs_sort_size);
}

#if USE_READLINE == 1
char * tab_completion_cmd (char * text, int state)
{
	static char * list[] = { "list", "list-sel", "list-unsel", "list-cat",
	                         "sort-name", "sort-size", "select", "unselect",
	                         "info", "files", "find", "sum", "df", "quit",
	                         "exit", "install", "pager", "list-ext",
	                         "import", "unimport", NULL } ;
	static int c1;
	char *s;

	if (!state) c1=0;

	for ( ; list[c1]; c1++)
		if ( strncmp(text, list[c1], strlen(text)) == 0 ) {
			s = malloc(strlen(list[c1])+1);
			strcpy(s, list[c1]); c1++; return s;
		}

	return NULL;
}

char * tab_completion_para (char * text, int state)
{
	static int c1,c2,c3;
	static char oldcat[200];
	char *s,textbuf[200];

	if (!state) { c1=0; c2=0; c3=0; oldcat[0]=0; }

	if ( strchr(text,'/') != NULL ) {
		for ( ; c1 < pkg_num; c1++) {
			sprintf(textbuf, "%s/%s",
				pkgs[c1].folder, pkgs[c1].name);
			if ( strncmp(text, textbuf, strlen(text)) == 0 ) {
				s=malloc(strlen(textbuf)+1);
				strcpy(s, textbuf); c1++; return s;
			}
		}
		return NULL;
	}

	for ( ; c1 < pkg_num; c1++)
		if ( strncmp(text, pkgs[c1].name, strlen(text)) == 0 ) {
			s=malloc(strlen(pkgs[c1].name)+1);
			strcpy(s, pkgs[c1].name); c1++; return s;
		}

	if (!sortbyname) qsort(pkgs,pkg_num,sizeof(package),&pkgs_sort_name);
	for ( ; c2<pkg_num; c2++) {
		if (!strcmp(oldcat,pkgs[c2].folder)) continue;
		strcpy(oldcat,pkgs[c2].folder);
		sprintf(textbuf,"%s/*",oldcat);
		if ( strncmp(text, textbuf, strlen(text)) == 0 ) {
			if (!sortbyname) qsort(pkgs,pkg_num,sizeof(package),
			                       &pkgs_sort_size);
			s=malloc(strlen(textbuf)+1);
			strcpy(s, textbuf); c2++; return s;
		}
	}
	if (!sortbyname) qsort(pkgs,pkg_num,sizeof(package),&pkgs_sort_size);

	for ( ; c3<epkg_num*2; c3++) {
		if (c3%2 == 0) strcpy(textbuf,epkgs[c3/2].name);
		else sprintf(textbuf,"%s/%s",
			epkgs[c3/2].folder,epkgs[c3/2].name);
		if ( strncmp(text, textbuf, strlen(text)) == 0 ) {
			s=malloc(strlen(textbuf)+1);
			strcpy(s, textbuf); c3++; return s;
		}
	}

	return NULL;
}

char ** tab_completion (char * text, int start, int end)
{
	if (!start) return (char**) completion_matches (text, tab_completion_cmd);
	return (char**)completion_matches (text, tab_completion_para);
}
#endif

int main(int argc, char ** argv)
{
	char * basedir;
	char * instdir;
	char * infodir;
	
	signal(SIGINT,SIG_IGN);
	
	/* command line options */
	if (argc != 3) usage("Abort: Wrong Nr. of Arguments!\n");
	basedir=argv[1]; if ( strrchr(basedir,'/') &&
		strrchr(basedir,'/')[1]==0 ) strrchr(basedir,'/')[0]=0;
	instdir=argv[2]; if ( strrchr(instdir,'/') &&
		strrchr(instdir,'/')[1]==0 ) strrchr(instdir,'/')[0]=0;
	
	/* Transfer infodir data (if needed) */
	if ( !strncmp(basedir,"ftp://",6) || !strncmp(basedir,"http://",7) )
	{
		char cmd[2000];
		printf("\nNet-Install: Downloading meta information "
		       "and overriding the infodir setting.\n");
		sprintf(cmd,"rm -rf %s/inst-tmp ; mkdir -p %s/inst-tmp ; cd "
		        "%s/inst-tmp ; wget -q -O - %s/info/info.tar.bz2 | "
		        "tar xf - --use-compress-program=bzip2",
		        instdir,instdir,instdir,basedir);
		system(cmd);
		infodir=malloc(strlen(instdir)+20);
		sprintf(infodir,"%s/inst-tmp",instdir);
	} else {
		infodir=malloc(strlen(basedir)+10);
		sprintf(infodir,"%s/info",basedir);
	}
	
	/* Read Package List */
	{
		char fn[200];
		char line[200];
		FILE *f; DIR *d;
		struct dirent *dent;
		char xo; int sel,rc;
		
		sprintf(fn,"%s/packages",infodir);
		if ( !(d=opendir(fn)) )
			{ perror("ERROR: Open infodir/packages"); return 1; }
		pkg_num=sel=0;
		while ( (dent=readdir(d)) ) {
			if ( !strcmp(dent->d_name,"rock-debug") ||
			     (dent->d_name[0] == '.') ) continue;
			/* open the file and read the last line */
			sprintf(fn,"%s/packages/%s",infodir,dent->d_name);
			if ((f=fopen(fn,"rt")) == NULL) {
			  printf ("ERROR: Can't open %s", fn);
			  continue;
			}
			while (fgets(line,200,f));
			fclose(f);
			rc=sscanf(line,"%c %*s %s",&xo,pkgs[pkg_num].folder);

			/* Skip TRANS.TBLs and others */
			if ( rc != 2 ) continue;
			if ( (xo != 'x' ) && (xo != 'o' ) &&
			     (xo != 'z' ) && (xo != 'e' ) ) continue;

			/* Handle ext packages */
			if ( xo == 'e' && pkgs[pkg_num].folder[0]=='.' ) {
				sscanf(line,"e %s",epkgs[epkg_num].folder);
				strcpy(epkgs[epkg_num].name,dent->d_name);
				epkgs[epkg_num].imported=0; epkg_num++;
				continue;
			}
	
			/* copy the package information */
			strcpy(pkgs[pkg_num].name,dent->d_name);
			pkgs[pkg_num].selected=(xo=='x');
			pkgs[pkg_num].size=pkg_size(pkgs[pkg_num].name,infodir);
			sprintf(pkgs[pkg_num].fullname,"%s/%s",
			        pkgs[pkg_num].folder,pkgs[pkg_num].name);
			sprintf(pkgs[pkg_num].filename,"base-pkgs/%s.tar.bz2",
			        pkgs[pkg_num].name);
			if (pkgs[pkg_num].selected) sel++;
			pkg_num++;
		}
		closedir(d);

		qsort(pkgs,pkg_num,sizeof(package),&pkgs_sort_name);
		qsort(epkgs,epkg_num,sizeof(ext_package),&epkgs_sort_name);
	}
	printf("\nROCK Linux " ROCKVER " Install Shell.\n\n");
	inst_sum(basedir,infodir,instdir); printf("\n");

#if USE_READLINE == 1
	rl_attempted_completion_function = (CPPFunction *)tab_completion;
#endif
	
	/* Main Loop */
	while (1) {
		char line[100];
		char cmd[100];
		char arg[100];
#if USE_READLINE == 1
		char *readline_dat;
#endif

		arg[0]=0; cmd[0]=0; line[0]=0;
#if USE_READLINE == 1
		readline_dat = readline("ROCK Install Shell> ");
		if (readline_dat) {
			strncpy(line,readline_dat,98);
			line[98]=0; strcat(line,"\n");
			if (readline_dat[0])
				add_history(readline_dat);
			free(readline_dat);
		} else
			line[0]=0;
#else
		printf("ROCK Install Shell> "); fflush(stdout);
		line[99]=0; fgets(line,99,stdin);
		if ( !isatty(0) ) fputs(line,stdout);
#endif
		while (line[0] && line[1] && (line[strlen(line)-2] == ' ' ||
		                              line[strlen(line)-2] == '\t'))
			{ line[strlen(line)-1]=0; line[strlen(line)-1]='\n'; }
		if ( !line[0] ) { fputs("\n",stdout); break; }
		sscanf(line,"%s %[^\n]",cmd,arg);
		if ( !cmd[0] ) continue;
		
		if ( !strcmp(cmd,"?") || !strcmp(cmd,"help") ||
		     !strcmp(cmd,"h") ) help();
		else if ( !strcmp(cmd,"list") ) list(-1,arg);
		else if ( !strcmp(cmd,"list-sel") ) list(1,arg);
		else if ( !strcmp(cmd,"list-unsel") ) list(0,arg);
		else if ( !strcmp(cmd,"list-cat") ) listcat();
		else if ( !strcmp(cmd,"list-ext") ) listext(arg);
		else if ( !strcmp(cmd,"imp") || !strcmp(cmd,"import") ) ext_import(arg,infodir);
		else if ( !strcmp(cmd,"unimp") || !strcmp(cmd,"unimport") ) ext_unimport(arg);
		else if ( !strcmp(cmd,"select") || !strcmp(cmd,"sel") ) sel_unsel(1,arg);
		else if ( !strcmp(cmd,"unselect") || !strcmp(cmd,"unsel") ) sel_unsel(0,arg);
		else if ( !strcmp(cmd,"info") ) pkg_info(arg,"packages",infodir);
		else if ( !strcmp(cmd,"files") ) pkg_info(arg,"flists",infodir);
		else if ( !strcmp(cmd,"find") ) find_file(arg,infodir);
		else if ( !strcmp(cmd,"sum") ) inst_sum(basedir,infodir,instdir);
		else if ( !strcmp(cmd,"df") ) print_df(infodir,instdir);
		else if ( !strcmp(cmd,"!") ) system(arg);
		else if ( !strcmp(cmd,"pager") ) sscanf(arg,"%d",&pg_max);
		else if ( !strcmp(cmd,"sort-name") ) { sortbyname=1;
			qsort(pkgs,pkg_num,sizeof(package),&pkgs_sort_name); }
		else if ( !strcmp(cmd,"sort-size") ) { sortbyname=0;
			qsort(pkgs,pkg_num,sizeof(package),&pkgs_sort_size); }
		else if ( !strcmp(cmd,"q") || !strcmp(cmd,"quit") ||
		          !strcmp(cmd,"exit") ) break;

		else if ( !strcmp(cmd,"setup") ) post_install(instdir);
		    /* undocumented command for retry a failed setup
			added by Nikolaus Filus */

		else if ( !strcmp(cmd,"install") ||
		          !strcmp(cmd,"inst")) {
			qsort(pkgs,pkg_num,sizeof(package),&pkgs_sort_name);
			install(basedir,instdir,infodir);
			printf("\nWould you like me to run the setup scripts now? (An alternative would be\n"
			       "to install another CD first, or to do this manually) (YES/no) ");
			
			/* hack by Valentin Ziegler */
			fflush(stdout);

			line[0]=line[99]=0;
			if (!fgets (line, 99, stdin))
			  {
			    printf("\n");
			    break;
			  }
			if (!isatty(0))
			  fputs(line,stdout);

			if (!line[0])
			  {
			    printf("\n");
			    break;
			  }

			if (line[0]!='n' && line[0]!='N')
			  post_install(instdir);

			break;
			
		}
		else printf("Invalid command. Type 'help' for help.\n");
		pg_ready();
	}

	/* Remove temp-infodir (if needed) */
	{
		char cmd[2000];
		sprintf(cmd,"%s/inst-tmp",instdir);
		if ( !strcmp(cmd,infodir) ) {
			sprintf(cmd,"rm -rf %s/inst-tmp",instdir);
			system(cmd);
		}
	}
	
	return 0;
}
