/******************************************************
 * Desktop ROCK Linux menu installer.                 *
 * Copyright 2001-2002 Valentin Ziegler and Ren Rebe *
 * Released under the terms of the Gerneral Public    *
 * License.                                           *
 ******************************************************/

/*
 * ChangeLog:
 * 2001: Valentin Ziegler
 *   * creation and some bug-fixes
 * 2002-02-08: Ren Rebe
 *   * added necessary comments and copy-right notice
 *   * converted all " . " into "."
 *   * fixed save-selection without filesname crash
 *   * fixed summery-info garbage text / crash
 *   * made the about dialog prettier
 */

#include <curses.h>
#include <signal.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <pty.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/wait.h>
#include <stdio.h>
#include <string.h>

#ifndef ROCKTEXT
#  define ROCKTEXT "dRock "
#endif

#ifndef MAINTITLE
#  define MAINTITLE "dRock install system v0.7"
#endif

#ifndef PROMT
#  define PROMT "ROCK Install Shell> "
#endif

#define LINES_MIN 18
#define COLS_MIN 50

#define BLOCK 1280

int return_code = 0;
FILE* logfile;


static void finish (int sig)
{
  fprintf (logfile, "\n\nterminating (returncode %d)\n", return_code);
  fclose (logfile);
  endwin ();
  exit (return_code);
}

void* smalloc (size_t size)
{
  void* tmp = malloc (size);
  if (!tmp)
    {
      fprintf (stderr, "FATAL: out of mem.\n");
      fprintf (logfile, "FATAL: out of mem.\n");
      return_code = -1;
      finish (0);
    }
  return tmp;
}

void* srealloc (void* ptr, size_t size)
{
  void* tmp = realloc (ptr, size);
  if (!tmp)
    {
      fprintf (stderr, "FATAL: out of mem.\n");
      fprintf (logfile, "FATAL: out of mem.\n");
      return_code = -1;
      finish (0);
    }
  return tmp;
}



/* =================================== pty stuff =================================== */

int com_channel;   /* fd of pty-master channel */
pid_t install_bin; /* pid of install_bin subprocess */

struct termios settings; /* virtual terminal settings */ 

typedef struct outputbuf_t outputbuf; /* install_bin output */

int error_by_death = 1; /* signal error in case install_bin is dead */

struct outputbuf_t
{
  char* buffer_start;
  char* output;
  long size;
  int dead;
};

/* forward decl */
void* text_box (char* title, outputbuf* data);

static void marshall (int sig)
{
  kill (install_bin, SIGINT);
}

int is_alive ()
{
  return (waitpid (install_bin, NULL, WNOHANG) == 0);
}

outputbuf* read_output (int cut_promt)
{
  fd_set channel;
  int cont = 1;
  char* tmp = (char*) smalloc (BLOCK);
  int pos = 0;
  int dead = 0;
  int do_cut = 0;
  int checksize = strlen (PROMT);
  int offs = 0;
  outputbuf* ret;

  do 
    {
      FD_ZERO (&channel);
      FD_SET (com_channel, &channel);
      select (FD_SETSIZE, &channel, NULL, NULL, NULL);

      if (FD_ISSET (com_channel, &channel)) 
	{
	  int res;
	  
	  if (BLOCK - offs == 0)
	    {
	      tmp = srealloc (tmp, pos + BLOCK);
	      offs = 0;
	    }	  

	  res = read (com_channel, tmp + pos, BLOCK - offs);
	  if (res <= 0)
	    {
	      dead = 1;
	      cont = 0;
	    }
	  else
	    {
	      pos += res;
	      offs += res;
	    }

	  if (pos >= checksize)
	    if (strncmp(PROMT, tmp + pos - checksize, checksize) == 0)
	      {
		do_cut = 1;
		cont = 0;
	      } 
	}

    }
  while (cont);

  for (cont == 0; cont < pos; cont++)
    fputc (tmp [cont], logfile);

  if (do_cut && cut_promt)
    pos -= checksize;

  ret = (outputbuf *) smalloc (sizeof (outputbuf));
  ret -> output = tmp;
  ret -> buffer_start = tmp;
  ret -> size = pos;
  ret -> dead = dead;

  return ret;
}

outputbuf* drop_cmd (char* cmd)
{
  int l =  strlen (cmd);
  outputbuf* ret;

  if (! is_alive () )
    {
      outputbuf *tmp = (outputbuf *) smalloc (sizeof (outputbuf));
      fprintf (logfile, "\n\nconnection with slave process broken\nemergency exit\n");
      tmp -> output = "connection with slave lost !\n";
      tmp -> buffer_start = tmp -> output; 
      tmp -> size = strlen (tmp -> output);
      text_box ("ERROR", tmp);
      finish (0);
    }

  write (com_channel, cmd, l);
  write (com_channel, "\n", 1);
  ret = read_output (1);
  ret -> output += (l + 1);
  ret -> size -= (l + 1);

  if (ret -> dead && error_by_death)
    {
      text_box ("ERROR", ret);
      finish (0);
    }

  return ret;
}


__inline__ void start_install_session (char* bin, char** args)
{
  struct winsize ws = {LINES, COLS, 0, 0};
  int tmp_install_bin;
  int tmp_com_channel;

  if ((tmp_install_bin = forkpty (&tmp_com_channel, NULL, &settings, &ws)) < 0)
    { 
      fprintf (logfile, " failed\n");
      fprintf (stderr, "cannot create pty pair\n");
      fprintf (logfile, "cannot create pty pair\n");
      return_code = -1;
      finish (0);
    }
  else if (tmp_install_bin == 0) {
    close (tmp_com_channel);
    execvp (bin, args);
    printf ("could not execute %s\n", bin);
    exit (1);
  }

  sleep (1);  

  install_bin = tmp_install_bin;
  com_channel = tmp_com_channel;
}

void kill_buf (outputbuf* buf)
{
  free (buf -> buffer_start);
  free (buf);
}



/* ================================= ncurses stuff ================================= */

/* ncurses attributes */ 

int a_backgrn;    /* background */
int a_border1;   /* window border 'shiny' */
int a_border2;   /* window border 'dark' */
int a_hilight;   /* highlighted menu item */
int a_hishort;   /* highlightet shortcut */
int a_lolight;   /* unselected menu item */
int a_loshort;   /* unselected shortcut */
int a_capture;   /* window titles */
int a_warning;   /* warnings, errormessages */

int draw_bg = 1; /* draw ROCKTEXT background ? */




typedef struct text_t text; /* this contains text which is
			       to be displayed in an environment 
			       with initialy unknown linewidth */

struct text_t
{
  char* words; /* terminated string, may contain '\n' for a forced new line; 
		  '&' for optional word division ('-') at end of line */
};


typedef struct menuitem_t menuitem;

struct menuitem_t
{
  char* item;
  text info;
};

text about_msg = {MAINTITLE"\n"
		  "(C) 2001-2002:\n\n"
		  "  Valentin Ziegler\n"
		  "    <ziegler@informatik.hu-berlin.de>\n"
		  "  Rene Rebe\n"
		  "    <rene@rocklinux.org>\n\n"
		  "An ASCII menu front&end to the ROCK LINUX INSTALL SHELL.\n"
		  "  (C) 2000 Clifford Wolf\n"};

text select_inf = {"Up/u, Down/d, PgUp/p, PgDn/n : navigation\n"
		   "Space : (un)se&lect   Enter/ESC/q : back to menu\n"   
		   "s : (un)select all  ? : info  f : package files "};

text menu_help = { "Menu navigation:\n\n"
		   "Up/u         : hilight previos item\n" 
		   "Down/d       : hilight next item\n"
		   "Enter/Space  : choose current item\n"
		   "ESC/q        : leave this menu\n\n"};

text text_box_help = { "Keystrokes:\n\n"
		       "Up/u     : scroll up one line\n"
		       "Down/d   : scroll down one line\n"
		       "Left/l   : scroll left\n"
		       "Right/r  : scroll right\n"
		       "PgDn/n   : scroll down 1/2 page\n"
		       "Pgup/p   : scroll up 1/2 page\n\n"
		       "to quit, press Enter/Space/ESC/q"};


menuitem mainmenu [6] = {

  {
    "package selection",
    {"Select which pack&ages should be in&stal&led"}
  },
   
  {
    "summary information",
    {"Show number of selected pack&ages and ex&pec&ted disk usage after in&stal&lation "}
  },

  {
    "console mode",
    {"Do it yourself : Manual pack&age sel&ec&tion via ROCK IN&STALL SHELL"}
  },

  {
    "install",
    {"Install "ROCKTEXT" and exit."}
  },

  {
    "about",
    {"About this program"}
  },

  {
    "quit",
    {"Exit without instal&lation"}
  }

};



/* generate ncurses atributes */
void generate_attributes ()
{
  if ( has_colors() )
    {
	/* pretty colored terminal */
	start_color ();
	init_pair (COLOR_BLUE, COLOR_BLUE, COLOR_BLUE);
	init_pair (COLOR_WHITE, COLOR_WHITE, COLOR_WHITE);
	init_pair (COLOR_BLACK, COLOR_BLACK, COLOR_WHITE);
	init_pair (COLOR_CYAN, COLOR_WHITE, COLOR_BLUE);
	init_pair (COLOR_MAGENTA, COLOR_YELLOW, COLOR_BLUE);
	init_pair (COLOR_YELLOW, COLOR_YELLOW, COLOR_WHITE);
	init_pair (COLOR_RED, COLOR_RED, COLOR_BLACK);

	a_backgrn = COLOR_PAIR (COLOR_BLUE) | A_BOLD;
	a_border1 = COLOR_PAIR (COLOR_WHITE) | A_BOLD; 
	a_border2 = COLOR_PAIR (COLOR_BLACK) | A_REVERSE;
	a_hilight = COLOR_PAIR (COLOR_CYAN) | A_BOLD;   
	a_hishort = COLOR_PAIR (COLOR_MAGENTA) | A_BOLD;
	a_lolight = COLOR_PAIR (COLOR_BLACK) | A_REVERSE;
	a_loshort = COLOR_PAIR (COLOR_YELLOW) | A_BOLD;
	a_capture = COLOR_PAIR (COLOR_YELLOW) | A_BOLD;
	a_warning = COLOR_PAIR (COLOR_RED) | A_BOLD;
    }
  else
    {
	/* ugly monochrome terminal  ...  Buahhrg :-( */
	a_backgrn = A_DIM;
	a_border1 = A_NORMAL;
	a_border2 = A_NORMAL;
	a_hilight = A_REVERSE;
	a_hishort = A_NORMAL;
	a_lolight = A_NORMAL;
	a_loshort = A_REVERSE;
	a_capture = A_BOLD;
	a_warning = A_BOLD;
    }
}

void init_ncurses ()
{
 // (void) signal (SIGINT, finish);      /* arrange interrupts to terminate */
 // (void) initscr ();      /* initialize the curses library */
  keypad (stdscr, TRUE);  /* enable keyboard mapping */
//  (void) nonl ();         /* tell curses not to do NL->CR/NL on output */
  (void) cbreak ();       /* take input chars one at a time, no wait for \n */
  (void) noecho ();       /* don't echo input */

  if (LINES < LINES_MIN || COLS < COLS_MIN)
    {
      fprintf (logfile, " failed\n");
      addstr ("Sorry, your terminal is too small...");
      return_code = -1;
      refresh (); getch ();
      finish (0);
    }

  generate_attributes ();
}


/* draws the screen background */
void draw_background ()
{
  char text[] = ROCKTEXT;
  int div = (COLS / strlen (text)) + 1;
  int y, x;

  attrset (a_backgrn);

  if (draw_bg)
    for (y = 0; y < LINES; y++)
      {	
	move (y, 0);
	for (x = 0; x < div; x++)
	  addstr (text);
      }
}



WINDOW* init_window (const char* title, const char* subtitle, int w, int h)
{
  WINDOW* retwin = newwin (h, w, (LINES - h) / 2, (COLS - w) / 2); 
  if (retwin == NULL)
    {
      attrset (a_warning);
      mvaddstr (5, 5, "Failed to create ncurses-window !");
      return_code = -1;
      refresh ();
      getch ();
      finish (0);
    }

  wborder(retwin, ACS_VLINE | a_border1, ACS_VLINE | a_border2, 
	  ACS_HLINE | a_border1, ACS_HLINE | a_border2, 
	  ACS_ULCORNER | a_border1, ACS_URCORNER | a_border2, 
	  ACS_LLCORNER | a_border1, ACS_LRCORNER | a_border2);

  if (title)
    { 
      wattrset (retwin, a_border1);  mvwaddch (retwin, 0, 4, ACS_RTEE); 
      wattrset (retwin, a_capture);  waddnstr (retwin, title, w - 8); 
      wattrset (retwin, a_border1);  waddch (retwin, ACS_LTEE);
    }

  if (subtitle)
    { 
      int l = strlen (subtitle);
      wattrset (retwin, a_border2);  mvwaddch (retwin, h - 1, (l > w - 4) ? 1 : (w - 3) - l, ACS_RTEE); 
      wattrset (retwin, a_capture);  waddnstr (retwin, subtitle, w - 4); 
      wattrset (retwin, a_border2);  waddch (retwin, ACS_LTEE);
    }

  return retwin;
}


WINDOW* init_sub_window (WINDOW* parent, const char* subtitle, int w, int h, int x, int y)
{
  WINDOW* retwin = derwin (parent, h, w, x, y); 
  if (retwin == NULL)
    {
      attrset (a_warning);
      mvaddstr (5, 5, "Failed to create ncurses-window !");
      return_code = -1;
      refresh ();
      getch ();
      finish (0);
    }

  wattrset (retwin, A_NORMAL);

  wborder(retwin, ACS_VLINE | a_border2, ACS_VLINE | a_border1, 
	  ACS_HLINE | a_border2, ACS_HLINE | a_border1, 
	  ACS_ULCORNER | a_border2, ACS_URCORNER | a_border1, 
	  ACS_LLCORNER | a_border2, ACS_LRCORNER | a_border1);

  if (subtitle)
    {
      wattrset (retwin, a_border2);  mvwaddch (retwin, 0, 4, ACS_RTEE); 
      wattrset (retwin, a_lolight);  waddnstr (retwin, subtitle, w - 8); 
      wattrset (retwin, a_border2);  waddch (retwin, ACS_LTEE);
    }

  return retwin;
}

int yes_no (char *question)
{
  int l = strlen (question);
  int w = (l > COLS - 2) ? COLS - 2 : (l < 15) ? 15 : l;
  int ch;
  int xpos = 1;
  WINDOW* mywin = init_window (NULL, NULL, w + 2, 4);

  wattrset (mywin, a_lolight);
  mvwaddnstr (mywin, 1, 1, question, w);
  for (ch = l; ch < w; ch++)
    mvwaddch (mywin, 1, ch + 1, ' ');
  for (ch = 0; ch < w; ch++)
    mvwaddch (mywin, 2, ch + 1, ' ');


  do
    {
      wattrset (mywin, (xpos) ? a_hilight : a_lolight);
      mvwaddstr (mywin, 2, 3, "<Yes>"); 
      wattrset (mywin, (!xpos) ? a_hilight : a_lolight);
      mvwaddstr (mywin, 2, 9, "<No>"); 
      wattrset (mywin, (xpos) ? a_hishort : a_loshort);
      mvwaddch (mywin, 2, 4, 'Y');
      wattrset (mywin, (!xpos) ? a_hishort : a_loshort);
      mvwaddch (mywin, 2, 10, 'N');
      wrefresh (mywin);    

      ch = getch ();
      switch (ch)
	{
	case KEY_LEFT:
	case KEY_RIGHT:
	  xpos = ! xpos;
	  break;

	case ' ':
	case '\n':
	  ch = xpos;
	  break;

	case 'y':
	case 'Y':
	  ch = 1;
	  break;

	case 27:
	case 'n':
	case 'N':
	  ch = 0;
	  break;

	default:
	  beep ();
	  ch = 2;
	  break;
	} 
    }
  while (ch > 1);

  delwin (mywin);

  return ch;
}


void help_box (const text* helptext);

void* text_box (char* title, outputbuf* data)
{
  WINDOW* mywin;
  int i;
  int cols = 0;
  int lines = 0;
  int last = 0;
  int ch;

  char** line = NULL;
  int* line_cols = NULL;

  for (i = 0; i < data -> size; i++)
    if (data -> output [i] == '\n')
      {
	if (lines % BLOCK == 0)
	  {
	    line = (char **) srealloc (line,  sizeof (char*) * (BLOCK + lines + 1));
	    line_cols = (int *) srealloc (line_cols, sizeof (int) * (BLOCK + lines + 1));
	    line [0] = data -> output;
	  }

	line_cols [lines] = i - last;
	if (line_cols [lines] > cols)
	  cols = line_cols [lines];
	lines ++;
	i ++;
	last = i;

	line [lines] = data -> output + i;
      }
  line_cols [lines] = i - last;
  if ( line_cols [lines] > 0)
    lines ++;
  {
    int xpos = 0;
    int ypos = 0;
    int w = (cols + 2 > COLS - 2) ? COLS - 2 : cols + 2;
    int h = (lines + 2 > LINES - 2) ? LINES - 2 : lines + 2;
    mywin = init_window (title, "? / h : help", w, h);

    if (cols + 2 > w)
      {
	wattrset (mywin, a_capture);
	mvwaddch (mywin, h - 1, (w / 2) - 1, ACS_LARROW);
	mvwaddch (mywin, h - 1, (w / 2), ACS_HLINE);
	mvwaddch (mywin, h - 1, (w / 2) + 1, ACS_RARROW);
      }
      

    if (lines + 2 > h)
      {
	wattrset (mywin, a_capture);
	mvwaddch (mywin, (h / 2) - 1, w - 1, ACS_UARROW );
	mvwaddch (mywin, (h / 2), w - 1, ACS_VLINE );
	mvwaddch (mywin, (h / 2) + 1, w - 1, ACS_DARROW );
      }
    
    do
      {
	wattrset (mywin, a_lolight);
	for (i = 1; i < h - 1; i++)
	  {
	    int cline = (i + ypos) - 1;
	    int fill;
	    if ((fill = line_cols [cline] - xpos) > 0)
	      {
		if (fill > w - 2)
		  fill = w - 2;
		mvwaddnstr (mywin, i, 1, line [cline] + xpos, fill);
	      }
	    else 
	      fill = 1;
	      
	    wmove (mywin, i, fill);
	    for (fill; fill < (w - 1); fill++)
	      waddch (mywin, ' ');
	  }

	wrefresh (mywin);
	ch = getch ();
	switch (ch)
	  {
	  case KEY_DOWN:
	  case 'd':
	    if (ypos < lines - h) 
	      ypos ++;
	    else
	      beep ();
	    break;

	  case KEY_NPAGE:
	  case 'n':
	    if (ypos < lines - h)
	      { 
		ypos += h / 2;
		if (ypos > lines - h)
		  ypos = lines - h;
	      }
	    else
	      beep ();
	    break;

	  case KEY_PPAGE:
	  case 'p':
	    if (ypos > 0)
	      { 
		ypos -= h /2;
		if (ypos < 0)
		  ypos = 0;
	      }
	    else
	      beep ();
	    break;

	  case KEY_UP:
	  case 'u':
	    if (ypos > 0) 
	      ypos --;
	    else
	      beep ();
	    break;

	  case KEY_RIGHT:
	  case 'r':
	    if (xpos - 2 < cols - w) 
	      xpos ++;
	    else
	      beep ();
	    break;

	  case KEY_LEFT:
	  case 'l':
	    if (xpos > 0) 
	      xpos --;
	    else
	      beep ();
	    break;

	  case 27:
	  case ' ':
	  case 'q':
	  case '\n':
	    ch = 0;
	    break;

	  case '?':
	  case 'h':
	    help_box (&text_box_help);
	    draw_background ();
	    refresh ();
	    redrawwin (mywin);	    
	    break;

	  default:
	    beep ();
	    ch = -1;
	  } 
      }
    while (ch != 0);
    delwin (mywin);
  }

  free (line);
  free (line_cols);
}



/* draws content of <text> struct in given rectangle */
void drawtext (WINDOW* win, const text* mtext, int col_start, int line_start, int col_max, int line_max)
{
  char* current_char = mtext -> words;
  int line = line_start;
  int col = col_start;

  wmove (win, line_start, col_start);

  while (*current_char != '\0')
    {
      if (col <= col_max)
	{
	  if (*current_char != '&') /* skip word-division */
	    if (*current_char == '\n')
	      {
		/* user newline */
		while (col++ <= col_max)
		  waddch (win, ' ');
	      }
	    else
	      {
		waddch (win, *current_char);
		col++;
	      }

	  current_char ++;
	}
      else
	{ /* line is full */
	  if (line == line_max)
	    { /* no space left */
	      mvwaddstr (win, line_max, col_max - 3, "[..]");
	      return;
	    }

	  if (*current_char != ' ')
	    current_char --; 

	  if (*current_char != '\n')
	    {
	      /* try to divide current word */
	      while (*current_char != ' ' && *current_char != '&')
		{
		  col --;
		  if (col < col_start)
		    return; /* emergency exit : word greater than line and no division possible */
		  
		  mvwaddch (win, line, col, ' ');
		  current_char --;
		}
	    }
	     
	  if (*current_char == '&')
	    mvwaddch (win, line, col, '-');
	  else
	    current_char++;
	    

	  line++;
	  col = col_start;
	  wmove (win, line, col);
	}
    }

  /* fill rectangle */
  while (line++ <= line_max)
    {
      while (col++ <= col_max)
	waddch (win, ' ');
       
      col = col_start;
      wmove (win, line, col);
    }
}



void draw_about_screen ()
{
  int ch;
  WINDOW* mywin; 
  
  mywin= init_window ("About", "Press Enter or Space", 45, 14);
  wattrset (mywin, a_lolight);
  drawtext (mywin, &about_msg, 1, 1, 43, 12);
  wrefresh (mywin);

  do ch = getch ();
  while (ch != '\n' && ch != ' '); 
  delwin (mywin);
}


void help_box (const text* helptext)
{
  WINDOW* win;
  char ch;
  draw_background ();
  win = init_window ("Help", "Press Enter or Space", 45, 14);
  drawtext (win, helptext, 1, 1, 43, 12);
  refresh ();
  wrefresh (win);
  do ch = getch ();
  while (ch != '\n' && ch != ' '); 
  delwin (win);
}


char* dialog (char* title) /* Query user to enter string. returns NULL when aborted */
{
  char* str = 0;
  int chars = 0;
  int buffsize = 0;
  int ch;
  int i;
  int w = COLS - 2;
  int cursor = 0;
  int offs = 0;

  WINDOW* mywin = init_window (title, "ESC or empty string to abort", COLS, 3);


  do
    {
      wattrset (mywin, a_hilight);
      if (chars < w)
	{
	  if (chars > 0)
	    mvwaddnstr (mywin, 1, 1, str, chars);
	  for (i = chars; i < w; i++)
	    mvwaddch (mywin, 1, i + 1, ' ');
	}
      else
	{
	  if (cursor > offs + w)
	    offs = cursor - w;
	  if (cursor < offs)
	    offs = cursor;

	  if (offs + w> chars)
	    offs = chars - w;
	  mvwaddnstr (mywin, 1, 1, str + offs, w);
	}

      wmove (mywin, 1, (cursor - offs) + 1);
      wrefresh (mywin);

      ch = getch ();
      switch (ch)
	{
	case KEY_LEFT:
	  if (cursor > 0)
	    cursor --;
	  else
	    beep ();
	  break;

	case KEY_RIGHT:
	  if (cursor < chars)
	    cursor ++;
	  else
	    beep ();
	  break;

	case KEY_BACKSPACE:
	  if (cursor > 0)
	    {
	      if (cursor < chars)
		memmove (str + cursor - 1, str + cursor, chars - cursor);
	      chars --;
	      cursor --;
	    }
	  else
	    beep ();
	  break;

	case KEY_DC:
	  if (cursor < chars)
	    {
	      memmove (str + cursor, str + cursor + 1, chars - cursor);
	      chars --;
	    }
	  else
	    beep ();
	  break;


	case 27:
	  if (buffsize)
	    free (str);
	  chars = 0;

	case '\n':
	  ch = 0;
	  break;

	default:
	  if (ch < 0x20 || ch > 0x7e) /* filter non-ascii characters and controll chars */ 
	    {
	      beep ();
	      break;
	    }

	  if (chars == buffsize)
	    {
	      str = realloc (str, chars + BLOCK);
	      buffsize += BLOCK;
	    }

	  if (cursor < chars)
	    memmove (str + cursor + 1, str + cursor, chars - cursor);
	  str [cursor] = (char) ch;
	  chars ++;
	  cursor ++;
	}

    }
  while (ch != 0);

  if (chars)
    {
      if (chars == buffsize)
	str = realloc (str, chars + 1);
      str [chars] = '\0';
    }

  delwin (mywin);

  return (chars) ? str : NULL;
}


int menu (char* title, char* above, menuitem* list, int items, int info_heigth, int preselection)
{
  int i;
  int w;
  int h;
  int menuspace;
  WINDOW* mainwin;
  WINDOW* itemwin;
  int ch;
  int ypos = preselection;

  {
    /* calculate window proportions */
    int max_itemchars = 0;
    int add = info_heigth + 4;
    int menuspace_max = LINES - add;

    for (i = 0; i < items; i++)
      {
	if (strlen (list [i].item) > max_itemchars)
	  max_itemchars = strlen (list [i].item);
      }

    w = (max_itemchars + 2 > COLS - 2) ? COLS - 2 : (max_itemchars + 2 < COLS / 2) ? COLS / 2 : max_itemchars + 2;
    menuspace = (items > menuspace_max) ? menuspace_max : items;
    h = menuspace + add;
  }

  mainwin = init_window (title, "? / h : help", w, h);
  itemwin = init_sub_window (mainwin, above, w - 2, menuspace + 2, 1, 1); 

  do 
    {
      int begin = (ypos - (menuspace / 2));
      if (begin < 0)
	begin = 0;
      if (begin + menuspace > items)
	begin = items - menuspace;

      for (i = 0; i < menuspace; i++)
	{
	  wattrset (itemwin, (i + begin == ypos) ? a_hilight : a_lolight);
	  mvwaddnstr (itemwin, i + 1, 1, list [i + begin].item, w - 3);
	  for (ch = strlen (list [i + begin].item) + 1; ch < w - 3; ch ++)
	    mvwaddch (itemwin, i + 1 , ch , ' ');
	}

      drawtext (mainwin, &(list [ypos].info), 1 ,menuspace + 3, w - 2, h - 2);
      wrefresh (mainwin);
      wrefresh (itemwin);

      ch = getch ();
	switch (ch)
	  {
	  case KEY_DOWN:
	  case 'd':
	    if (ypos < items - 1) 
	      ypos ++;
	    else
	      beep ();
	    break;

	  case KEY_UP:
	  case 'u':
	    if (ypos > 0) 
	      ypos --;
	    else
	      beep ();
	    break;

	  case 27:
	  case 'q':
	    ypos = items - 1;

	  case '\n':
	  case ' ':
	    ch = 0;
	    break;

	  case '?':
	  case 'h':
	    help_box (&menu_help);
	    draw_background ();
	    refresh ();
	    redrawwin (mainwin);
	    redrawwin (itemwin);
	    break;

	  default:
	    beep ();
	    ch = -1;
	  }
    }

  while (ch != 0);

  delwin (itemwin);
  delwin (mainwin);
  return ypos;
}

char* pkgcount ()
{
  outputbuf* buf = drop_cmd ("sum");
  char* end;
  char* begin = buf -> output;
  int i = 0;
  char* str;

  while (i < 4)
    if (*(begin++) == '\n')
      i++;

  end = begin + 1;
  while (*(end++) != '\n');
 
  str = (char *) smalloc ((end - begin) + 2);
  strncpy(str, begin, end - begin);
  str [end - begin] = '\n';
  str [1 + end - begin] = '\0';
  kill_buf (buf);
  return str;
}

void draw_summary ()
{
  outputbuf* tmp = drop_cmd ("df");
  char* newdata = pkgcount ();
  int l = strlen (newdata);
  newdata = srealloc (newdata, l + tmp -> size + 1);
  memcpy (newdata + l, tmp -> output, tmp -> size);
  tmp -> size += l;
  free (tmp -> buffer_start);
  tmp -> buffer_start = newdata;
  tmp -> output = newdata;
  refresh (); 
  text_box ("summary", tmp);
  kill_buf (tmp);
}


void pkg_selection (char* matchstr)
{
  char* cmd = (char *) smalloc (strlen (matchstr) + 6);
  outputbuf* tmp;
  int files = 0;
  char** filelist = NULL;
  char** sizes = NULL;
  int* selected = NULL;
  int pos = 0;
  char file [256];
  int allsel = 1; 

  sprintf (cmd, "list %s", matchstr);
  tmp = drop_cmd (cmd);
  free (cmd);

  tmp -> output [tmp -> size] = '\0'; /* cut promt */

  do 
    {
      while (tmp -> output [pos] != 'o' && tmp -> output [pos] != 'x')
	{
	  pos++;
	  if (pos >= tmp -> size)
	    break;
	}
      
      if (pos >= tmp -> size)
	break;

      if (files % BLOCK == 0)
	{
	  filelist = (char **) srealloc (filelist, sizeof (char*) * (BLOCK + files));
	  sizes = (char **) srealloc (sizes, sizeof (char*) * (BLOCK + files));
	  selected = (int *) srealloc (selected, (BLOCK + files) * sizeof (int));
	}

      selected [files] = (tmp -> output [pos] == 'x');
      allsel = (allsel && selected [files]);
      pos++;
      sizes [files] = (char *) smalloc (6);
      sscanf (tmp -> output + pos, "%[^M]M", sizes [files]);
      pos += 7;
      sscanf (tmp -> output + pos, "%s", file);
      filelist [files] = strdup (file);
      pos += strlen (filelist [files]) + 1;
      files++;
    }
  while (pos < tmp -> size);

  if (files == 0)
    {
      outputbuf *tmp2 = (outputbuf *) smalloc (sizeof (outputbuf));
      tmp2 -> output = strdup ("no packages matching this pattern.\n");
      tmp2 -> buffer_start = tmp2 -> output; 
      tmp2 -> size = strlen (tmp2 -> output);
      text_box ("Sorry...", tmp2);
      kill_buf (tmp2);
    }
  else 
    {
      int w = COLS;
      int h = (files + 8 > LINES) ? LINES : files + 8;
      int ypos = 0;
      int space = h - 8;
      int ch;
      char* count = pkgcount ();
      WINDOW* mainwin = init_window ("package selection", NULL, w, h);
      WINDOW* itemwin = init_sub_window (mainwin, NULL, w - 2, h - 6, 1, 1); 

      drawtext (mainwin, &select_inf, 1, space + 3, w - 2, h - 2);

      draw_background ();
      refresh (); 

      do
	{
	  int begin = (ypos - ( space / 2));
	  int i;
	  outputbuf* tmp2;

	  if (begin < 0)
	    begin = 0;
	  if (begin + space > files)
	    begin = files - space;

	  for (i = 0; i < space; i++)
	    {
	      wattrset (itemwin, (i + begin == ypos) ? a_hilight : a_lolight);
      	      mvwaddstr (itemwin, i + 1, 1, (selected [i + begin])? "[x] " : "[ ] ");
	      mvwaddnstr (itemwin, i + 1, 5, filelist [i + begin], w - 8);
	      for (ch = strlen (filelist [i + begin]) + 5; ch < w - 12; ch ++)
		mvwaddch (itemwin, i + 1 , ch , ' ');
	      mvwaddnstr (itemwin, i + 1, w - 12, sizes [i + begin], 6);
	      mvwaddstr (itemwin, i + 1 , w - 6 , " Mb");
	    }

	  wattrset (mainwin, a_lolight);
	  mvwaddnstr (mainwin, h - 2 , 1, count, strlen (count) - 2);
	  mvwaddstr (mainwin, h - 2 , strlen (count) - 2 , "   ");
	  wrefresh (mainwin);
	  wrefresh (itemwin);

	  ch = getch ();
	  switch (ch)
	    {
	    case KEY_DOWN:
	    case 'd':
	      if (ypos < files - 1) 
		ypos ++;
	      else
		beep ();
	      break;
	      
	    case KEY_UP:
	    case 'u':
	      if (ypos > 0) 
		ypos --;
	      else
		beep ();
	      break;

	  case KEY_NPAGE:
	  case 'n':
	    if (ypos < files - 1)
	      { 
		ypos += h / 2;
		if (ypos > files - 1)
		  ypos = files - 1;
	      }
	    else
	      beep ();
	    break;

	  case KEY_PPAGE:
	  case 'p':
	    if (ypos > 0)
	      { 
		ypos -= h / 2;
		if (ypos < 0)
		  ypos = 0;
	      }
	    else
	      beep ();
	    break;

	    case ' ':
	      cmd = (char *) smalloc (strlen (filelist [ypos]) + 8);
	      sprintf (cmd, (selected [ypos])? "unsel %s" : "select %s", filelist [ypos]);
	      kill_buf (drop_cmd (cmd));
	      free (cmd);
	      selected [ypos] = (! selected [ypos]);
	      free (count);
	      count = pkgcount ();
	      if (ypos < files - 1)
		ypos ++;
	      break;

	    case '?':
	      sscanf (filelist [ypos], "%*[^/]/%s", file);
	      cmd = (char *) smalloc (strlen (file) + 6);
	      sprintf (cmd, "info %s", file);
	      tmp2 = drop_cmd (cmd);
	      free (cmd);
	      text_box ("package info", tmp2);
	      kill_buf (tmp2);
	      redrawwin (mainwin);
	      break;

	    case 'f':
	      sscanf (filelist [ypos], "%*[^/]/%s", file);
	      cmd = (char *) smalloc (strlen (file) + 7);
	      sprintf (cmd, "files %s", file);
	      tmp2 = drop_cmd (cmd);
	      free (cmd);
	      text_box ("package files", tmp2);
	      kill_buf (tmp2);
	      redrawwin (mainwin);
	      break;

	    case 's':
	      cmd = (char *) smalloc (strlen (matchstr) + 8);
	      sprintf (cmd, (allsel)? "unsel %s" : "select %s", matchstr);
	      kill_buf (drop_cmd (cmd));
	      free (cmd);
	      allsel = (! allsel);
	      for (pos = 0; pos < files; pos++)
		selected [pos] = allsel;
	      free (count);
	      count = pkgcount ();
	      break;

	    case 27:
	    case 'q':
	    case '\n':
	      ch = 0;
	      break;

	    default:
	      beep ();
	      ch = -1;
	    }
	}
      while (ch != 0);
      
      delwin (itemwin);
      delwin (mainwin);
    }

  kill_buf (tmp);

  for (pos = 0; pos < files; pos++)
    {
      free (filelist [pos]);
      free (sizes [pos]);
    }

  free (filelist);
  free (sizes);
  free (selected);
}


void select_menu ()
{
  outputbuf* tmp = drop_cmd ("list-cat");
  menuitem itemlist [128];
  int items = 0;
  char cat [128];  
  const char *infotxt = "list packages matching ";
  const char *itemtxt = "list ";
  int selection = 0;

  tmp -> output [tmp -> size] = '\0'; /* cut promt */

  while (sscanf (tmp -> output, "%s", &cat) > 0)
    {
      int l = strlen (cat);
      itemlist [items].item = (char*) smalloc (l + strlen (itemtxt) + 1);
      itemlist [items].info.words = (char*) smalloc (l + strlen (infotxt) + 2);
      sprintf (itemlist [items].item, "%s%s", itemtxt, cat);
      sprintf (itemlist [items].info.words, "%s%s\n", infotxt, cat);
      items ++;
      while (*(tmp -> output++) == ' ');
      tmp -> output += l;
    }

  itemlist [items] = (menuitem) {"list custom pattern", {"list packages matching search pattern"}};
  itemlist [items + 1] = (menuitem) {"load selection", {"read selection file from a file"}};
  itemlist [items + 2] = (menuitem) {"save selection", {"save current selection to a file"}};
  itemlist [items + 3] = mainmenu [1];
  itemlist [items + 4] = (menuitem) {"back", {"back to mainmenu"}};
  items += 5;

  do
    {
      draw_background ();
      refresh (); 

      selection = menu ("Package selection", NULL, itemlist, items, 2, selection);
      
      /* a package category for selection */
      if (selection < items - 5)
	pkg_selection (itemlist [selection].item + strlen (itemtxt));
      
      /* match custom pattern */
      if (selection == items - 5)
	{
	  char* pattern = dialog ("enter search pattern");
	  if (pattern)
	    {
	      pkg_selection (pattern);
	      free (pattern);
	    }
	}
      
      /* load list */
      if (selection == items - 4)
	{
	  char* filename = dialog ("enter filename");
	  if (filename)
	    {
	      FILE* src = fopen (filename, "r");
	      if (src)
		{
		  char file [256];
		  strcpy (file, "select     ");
		  if (!yes_no ("merge with currently selected files ?"))
		    kill_buf (drop_cmd ("unsel *"));
		  while (fgets (file + 7, 249, src) != NULL)
		    if (strlen (file) > 7)
		      {
			file [strlen (file) - 1] = '\0'; /* overwrite tailing newline character */
			kill_buf (drop_cmd (file));
		      }
		  close (src);
		}
	      else
		{
		  outputbuf error;
		  char* errstr = "failed to open ";
		  error.output = smalloc (strlen (errstr) + strlen (filename) + 3);
		  sprintf (error.output, "%s %s \n", errstr, filename);
		  error.size = strlen (error.output);
		  text_box ("ERROR", &error);
		  free (error.output);
		}
	    }
	}
      
      /* save list */
      if (selection == items - 3)
	{
	  char* filename = dialog ("enter filename");
	  if (filename)
	    {
	      struct stat info;
	      int do_create = 0;
	      
	      outputbuf error;
	      char* errstr = "can not create/access file";
	      error.output = smalloc (strlen (errstr) + strlen (filename) + 3);
	      sprintf (error.output, "%s %s \n", errstr, filename);
	      error.size = strlen (error.output);
	      
	      if (stat (filename, &info))
		{
		  if (errno == ENOENT)
		    do_create = yes_no ("File does not exist. Create ?");
		  else
		    text_box ("ERROR", &error);
		}
	      else
		do_create = yes_no ("File exists. Overwrite ?");
	      
	      if (do_create)
		{
		  FILE* dump = fopen (filename, "w+");
		  if (dump)
		    {
		      int pos = 0;
		      outputbuf* tmp = drop_cmd ("list-sel");
		      char file [256];

		      while (pos < tmp -> size)
			{
			  while (tmp -> output [pos ++] != 'M')
			    if (pos >= tmp -> size) 
			      break;
			  if (pos < tmp -> size) 
			    {
			      pos ++;
			      sscanf (tmp -> output + pos, "%s", file);
			      fprintf (dump, "%s\n", file);
			      pos += strlen (file) + 1;
			    }
			}
		      
		      kill_buf (tmp);
		      fclose (dump);
		    }
		  else
		    text_box ("ERROR", &error);
		}
	      
	      free (filename);
	      free (error.output);
	    }
	}
      
      /* summary information */
      if (selection == items - 2)
	draw_summary ();
	
    }
  while (selection != items - 1);

  for (selection = 0; selection < items - 5; selection ++)
    {
      free (itemlist [selection].item);
      free (itemlist [selection].info.words);
    }
}



void console_mode ()
{
  int ch;
  char input;
  fd_set channel;

  kill_buf (drop_cmd ("pager 15"));

  fprintf (logfile, "\n/// Entering console mode; logging turned off. ///\n");

  def_prog_mode();           /* save current tty modes */
  endwin();                  /* restore original tty modes */

  (void) cbreak ();       /* take input chars one at a time, no wait for \n */
  (void) noecho ();       /* don't echo input */

  write (0, "\nPress ESC to return...\nxx", 24);

  (void) signal (SIGINT, marshall);      /* send interrupts to slave process */

  write (com_channel, "\n", 1);

  do 
    {
      FD_ZERO (&channel);
      FD_SET (com_channel, &channel);
      FD_SET (0, &channel);
      if (select (FD_SETSIZE, &channel, NULL, NULL, NULL) > 0)
	{
	  ch = 0;
	  if (FD_ISSET (com_channel, &channel)) 
	    {
	      if (read (com_channel, &ch, 1) == 1)
		write (0, &ch, 1);
	      else
		ch = EOF;
	    }
	
	  if (FD_ISSET (0, &channel)) 
	    {
	      if (read (0, &input, 1) == 1)
		if (input != 27)
		  write (com_channel, &input, 1);
	    }      
	}
      else
	{
	  ch = 77;
	  input = 88;
	}
    } while (ch != EOF && input != 27);

  if (ch == EOF)
    {
      fprintf (logfile, "\n/// user exit in console mode ///\n");
      finish (0);
    }

  (void) signal (SIGINT, finish);      /* arrange interrupts to terminate */

  { /* uggly resync... wait for 12 v's... does anyone have a better idea ? */ 
    int vs = 0;
    const char* resync = "list vvvvvvvvvvvv\n";

    write (0, "\nresyncing....\nxxx", 15);
    kill (install_bin, SIGINT); /* quit pagers n stuff */
    sleep (1);
    write (com_channel, resync, strlen (resync));

    do 
      {
	FD_ZERO (&channel);
	FD_SET (com_channel, &channel);
	select (FD_SETSIZE, &channel, NULL, NULL, NULL);
	
	ch = 0;

	if (FD_ISSET (com_channel, &channel)) 
	  {
	    if (read (com_channel, &ch, 1) == 1)
	      vs = (ch == 'v') ? vs + 1 : 0;
	    else
	      ch = EOF;
	  }
      } while (ch != EOF && vs < 12);
    if (ch == EOF)
      {
	fprintf (logfile, "\n/// exit in console mode ///\n");
	finish (0);
      }

    kill_buf (read_output (1));
  }

  fprintf (logfile, "/// leaving console mode; logging turned on. ///\n");

  kill_buf (drop_cmd ("pager 0"));

  addstr("returned.\n");     /* prepare return message */
  refresh ();                 /* restore save modes, repaint screen */
}


void do_install ()
{
  int ch;
  char input;
  fd_set channel;

  kill_buf (drop_cmd ("pager 15"));

  fprintf (logfile, "\n/// starting installation in console mode; logging turned off. ///\n");

  def_prog_mode();           /* save current tty modes */
  endwin();                  /* restore original tty modes */

  (void) cbreak ();       /* take input chars one at a time, no wait for \n */
  (void) noecho ();       /* don't echo input */

  (void) signal (SIGINT, marshall);      /* send interrupts to slave process */

  write (com_channel, "install\n", 8);

  do 
    {
      FD_ZERO (&channel);
      FD_SET (com_channel, &channel);
      FD_SET (0, &channel);
      if (select (FD_SETSIZE, &channel, NULL, NULL, NULL) > 0)
	{
	  ch = 0;

	  if (FD_ISSET (com_channel, &channel)) 
	    {
	      if (read (com_channel, &ch, 1) == 1)
		write (0, &ch, 1);
	      else
		ch = EOF;
	    }

	  if (FD_ISSET (0, &channel)) 
	    {
	      if (read (0, &input, 1) == 1)
		write (com_channel, &input, 1);
	    }      
	}
      else 
	{
	  ch = 66;
	  input = 77;
	}
    } while (ch != EOF);

  fprintf (logfile, "\ninstallation finished.\n");
  finish (0);
}


void main_menu ()
{
  int selection = 0;
  do
    {
      draw_background ();
      refresh (); 
      selection = menu (MAINTITLE, "choose action", mainmenu, 6, 2, selection);

      switch (selection)
	{
	case 0:
	  select_menu ();
	  break;
	case 1:
	  draw_summary ();
	  break;
	case 2:
	  console_mode ();
	  break;
	case 3:
	  if (yes_no ("start installation"))
	    do_install ();
	  break;
	case 4:
	  draw_about_screen ();
	  break;
	case 5:
	  if (yes_no ("really quit ?"))
	    selection = 6;
	  break;
	}
			
    } 
  while (selection != 6);
}


/* ============================================================ */

main (int argc, char *argv[])
{
  outputbuf* buf;

  if (argc < 4)
    {
      fprintf (stderr, "usage: %s <logfile> <intstall bin> <src dir> <trg dir>\n", argv [0]);
      exit (-1);
    }

  if ((logfile = fopen (argv [1], "w+")) == NULL)
    {
      fprintf (stderr, "usage: %s <logfile> <intstall bin> <src dir> <trg dir>\n", argv [0]);
      exit (-1);
    }

  fprintf (logfile, "%s logging to %s\n", argv [0], argv [1]);
  fprintf (logfile, "getting terminal io settings ...");

  if (tcgetattr (0, &settings))
    {    
      fprintf (stderr, "cannot get terminal io settings...\n"); 
      fprintf (logfile, " failed\n");
      exit (-1);
    }

  fprintf (logfile, " done\ninitializing ncurses ..."); 

  (void) signal (SIGINT, finish);      /* arrange interrupts to terminate */
  (void) initscr (); 

  fprintf (logfile, " done\nstarting %s in a virtual session ...", argv [2]); 
  
  start_install_session (argv [2], argv + 2);

  fprintf (logfile, " done\nconfiguring ncurses ..."); 

  init_ncurses ();

  fprintf (logfile, " done\n\n=========== %s output ============\n\n", argv [2]);

  draw_background ();
  refresh ();
  
  buf = read_output (1);
  if (buf -> dead)
    {
      text_box ("ERROR", buf);
      finish (0);
    }
  text_box ("installation info", buf);
  kill_buf (buf);

  kill_buf (drop_cmd ("pager 0"));
  kill_buf (drop_cmd ("import *"));

  main_menu ();

  error_by_death = 0;
  drop_cmd ("quit");
  finish (0);               /* we're done */
}

